class Fft extends Surf {

/* One and two-dimensional FFT routines.  Test programs for all routines
   follow the routines. */

static double sdat[] = new double[NMSSTP + 1];
static double tdat[] = new double[NMTSTP + 1];
static double stdat[][] = new double[NMSSTP][NMTSTP];
static double imagdat[][] = new double[NMSSTP][NMTSTP];
static double trnsfrm_real[][] = new double[NMSSTP][NMTSTP];
static double trnsfrm_imag[][] = new double[NMSSTP][NMTSTP];

// ---------------------------------------------------------------------

static void fft_(int k, int nns, int nnt, double sl, double tl,
   double indat[][], double trnsfrm[][][][]) {

/* Returns the one or two-dimensional FFT of the "nns" by "nnt"
   values in "data." */

int i, j, ns, nt;
double deltas, deltat, real, imag, angle, ri, rj, a, b, c, d, realval, f1,
   f2;

if (spatial && !temporal) {

   /* Spatial-only.  Covariograms and cross-covariogram are even
      functions so that the transform is real-valued.  Hence, the
      imaginary part of the transform should be small. */

   ns = nns / 2;
   deltas = sl / ((double) ns);

   for (i = 0; i < nns; ++i) sdat[i + 1] = indat[i][0];
   realft_(sdat, ns);

   /* Spatial-lag shift back "sl" units.  See Brigham (1974, p. 37).
      It is assumed that an even function has been sampled nns times over
      the interval (-sl, sl).  Therefore, the sampling interval,
      delta = (2 * sl) / nns = sl / n where n = nns / 2.

      Let Hs(m) be the transform returned by realft_() (which is assumming
      the sampling interval is (0, 2 * sl)).  Hs(0) is always real-valued.
      Thereafter, pairs of real, complex components are returned for only
      the positive frequency values of 1, ..., ns.

      The sl-shifted transform is:
      H(m) = H(f) = exp(i * 2 * Math.PI * f * sl) * Hs(f)
		  = exp(i * 2 * Math.PI * (m / (delta * nns)) * sl) * Hs(f)
		  = exp(i * Math.PI * m) Hs(f) (because delta = 2 * sl / nns)
	          = (ac - bd) + i(ad + bc)
      where angle = Math.PI * m,
      where a = Math.cos(angle),
	    b = Math.sin(angle),
	    c = real(Hs(m)) = sdat[2 * i],
	    d = imag(Hs(m)) = sdat[2 * i + 1] */

   trnsfrm[k][0][0][0] = sdat[1] * deltas;
   trnsfrm[k][0][0][1] = 0.;
   for (i = 2; i <= ns; ++i) {
      angle = PI * ((double) (i - 1));
      real = sdat[2*i-1] * Math.cos(angle);
      real -= sdat[2*i] * Math.sin(angle);
      imag = sdat[2*i-1] * Math.sin(angle);
      imag += sdat[2*i] * Math.cos(angle);
      trnsfrm[k][i - 1][0][0] = real * deltas;
   }
   angle = PI * ((double) ns);
   real = sdat[2] * Math.cos(angle);
   imag = sdat[2] * Math.sin(angle);
   trnsfrm[k][ns][0][0] = real * deltas;

} else if (!spatial && temporal) {

   /* Temporal-only.  The covariograms are even functions but the
      cross-covariogram may by neither even nor odd.  Hence, the
      transform may be complex-valued.  Note that frequency zero
      is always real. */

   nt = nnt / 2;
   deltat = tl / (double) nt;

   for (i = 0; i < nnt; ++i) {
      tdat[i + 1] = indat[0][i];
      // if (k == 2) printf_("fft: " + i + " " + tdat[i+1]);
   }
   realft_(tdat, nt);

   /* Temporal-lag shift back "tl" units.  See Brigham (1974, p. 37)
      and comments above. */

   trnsfrm[k][0][0][0] = tdat[1] * deltat;
   trnsfrm[k][0][0][1] = 0.;

   for (i = 2; i <= nt; ++i) {
      angle = PI * ((double) (i - 1));
      real = tdat[2*i-1] * Math.cos(angle);
      real -= tdat[2*i] * Math.sin(angle);
      imag = tdat[2*i-1] * Math.sin(angle);
      imag += tdat[2*i] * Math.cos(angle);
      trnsfrm[k][0][i - 1][0] = real * deltat;
      trnsfrm[k][0][i - 1][1] = imag * deltat;
   }
   angle = PI * ((double) nt);
   real = tdat[2] * Math.cos(angle);
   imag = tdat[2] * Math.sin(angle);
   trnsfrm[k][0][nt][0] = real * deltat;
   trnsfrm[k][0][nt][1] = imag * deltat;

   // if (k == 2) {
   //   for (i = 0; i <= nt; ++i) {
   //      printf_("fftfreq: " + i + " " + trnsfrm[k][0][i][0]);
   //   }
   // }

} else if (spatial && temporal) {

   // Spatio-temporal.  Returns the space and time back-shifted transform.

   ns = nns / 2;
   nt = nnt / 2;
   deltas = 2. * sl / ((double) (nns - 1));
   deltat = 2. * tl / ((double) (nnt - 1));

   for (i = 0; i < nns; ++i) {
      for (j = 0; j < nnt; ++j) {
	 stdat[i][j] = indat[i][j];
	 imagdat[i][j] = 0.;
      }
   }

   ff2d_(0, nns, nnt, deltas, deltat, stdat, imagdat, trnsfrm_real,
      trnsfrm_imag);

   for (i = 1; i <= ns; ++i) {
      ri = (double) (i - 1);
      f1 = PIT2 * ri / (((double) nns) * deltas);
      for (j = 1; j <= nt; ++j) {
         rj = (double) (j - 1);
         f2 = PIT2 * rj / (((double) nnt) * deltat);

         /* Let Hs(ms,mt) be the transform returned by ff2d (which is
            assuming the sampling interval is ((0, 2 * sl),(0, 2 * tl))).

            (ms, mt) is the integer pair corresponding to the frequency pair
            (fs, ft).  Note that fs = ms / (deltas * nns) and
                                 ft = mt / (deltat * nnt).

            The (sl, tl)-shifted transform is:

            H(fs,ft) = exp(i * PIT2 * (fs * sl + ft * tl)) * Hs(fs,ft)

            and hence

            H(ms,mt) = exp(i * PIT2 * (sl * (ms / (deltas * nns)) +
                       tl * (mt / (deltat * nnt))) * Hs(ms,mt).

            Since sl = nns * deltas / 2 and tl = nnt * deltat / 2, the above is
       
            H(ms,mt) = exp(i * PI * (ms + mt)) * Hs(ms,mt).
                     = (ac - bd) + i(ad + bc)

            where
            a = cos(angle), b = sin(angle),
            c = real(Hs(ms,mt)), d = imag(Hs(ms,mt)).
            and angle = PI * (ms + mt) */

         angle = PI * (ri + rj);
         a = Math.cos(angle);
         b = Math.sin(angle);
         c = deltas * deltat * trnsfrm_real[i - 1][j - 1];
         d = deltas * deltat * trnsfrm_imag[i - 1][j - 1];
         realval = a * c - b * d;
         imag = a * d + b * c;
	 trnsfrm[k][i - 1][j - 1][0] = realval;
	 trnsfrm[k][i - 1][j - 1][1] = imag;
      }
   }
}
}

// ---------------------------------------------------------------------

static int four1_(double data[], int nn) {

// Fast Fourier Transform.  Adapted from Numerical Recipes, 1992, p. 507.

int n, mmax, m, j, istep, i;
double wtemp, wr, wpr, wpi, wi, theta, tempr, tempi;

n = nn<<1;
j = 1;
for (i = 1; i < n; i+=2) {
   if (j > i) {
      swap_(data, j, i);
      swap_(data, j + 1, i + 1);
   }
   m = n>>1;
   while (m >= 2 && j > m) {
      j -= m;
      m >>= 1;
   }
   j += m;
}

mmax = 2;
while (n > mmax) {
   istep = mmax<<1;
   theta = PIT2 / ((double) mmax);
   wtemp = Math.sin(.5 * theta);
   wpr = -2. * wtemp * wtemp;
   wpi = Math.sin(theta);
   wr = 1.;
   wi = 0.;
   for (m = 1; m < mmax; m += 2) {
      for (i = m; i <= n; i += istep) {
	 j = i + mmax;
	 tempr = wr * data[j] - wi * data[j + 1];
	 tempi = wr * data[j + 1] + wi * data[j];
	 data[j] = data[i] - tempr;
	 data[j + 1] = data[i + 1] - tempi;
	 data[i] += tempr;
	 data[i + 1] += tempi;
      }
      wr = (wtemp = wr) * wpr - wi * wpi + wr;
      wi = wi * wpr + wtemp * wpi + wi;
   }
   mmax = istep;
}

return 0;
}

// -------------------------------------------------------------------

static void realft_(double data[], int n) {

/* Calculates the Fourier transform of a set of n real-valued data
   points, from Numerical Recipes, p. 513. */

int i, i1, i2, i3, i4, n2p3;
double c1 = 0.5, c2, h1r, h1i, h2r, h2i, wr, wi, wpr, wpi, wtemp, theta;

theta = Math.PI / (double) n;
c2 = -.5;
four1_(data, n);
wtemp = Math.sin(.5 * theta);
wpr = -2. * wtemp * wtemp;
wpi = Math.sin(theta);
wr = 1. + wpr;
wi = wpi;
n2p3 = 2 * n + 3;
for (i = 2; i <= n / 2; i++) {
   i4 = 1 + (i3 = n2p3 - (i2 = 1 + (i1 = i + i - 1)));
   h1r = c1 * ( data[i1] + data[i3]);
   h1i = c1 * (data[i2] - data[i4]);
   h2r = -c2 * (data[i2] + data[i4]);
   h2i = c2 * (data[i1] - data[i3]);
   data[i1] = h1r + wr * h2r - wi * h2i;
   data[i2] = h1i + wr * h2i + wi * h2r;
   data[i3] = h1r - wr * h2r + wi * h2i;
   data[i4] = -h1i + wr * h2i + wi * h2r;
   wr = (wtemp = wr) * wpr - wi * wpi + wr;
   wi = wi * wpr + wtemp * wpi + wi;
}
data[1] = (h1r = data[1]) + data[2];
data[2] = h1r - data[2];
}

// -------------------------------------------------------------------

static void swap_(double vector[], int pos1, int pos2) {

// Swaps vector[pos1] with vector[pos2].

double temp;

temp = vector[pos1];
vector[pos1] = vector[pos2];
vector[pos2] = temp;
}

// -----------------------------------------------------------------------

static void tst1dfft_() {

/* Test of 1-dimensional Fast Fourier Transform.  Note that 2^10=1024.
   To run this test, set NMSSTP, NMTSTP in Id.java the value of nn,
   below.  Tests the function .5exp(t - l), l = 6.  With l=0, the fourier
   transform is 1/(1+4pi^2f^2) (Brigham, E. O. (1974) "The Fast Fourier
   Transform," Prentice Hall, Inc., p. 27). */

int i, n, nn = 32;
double data[] = new double[nn + 1];
double ximag[] = new double[nn + 1];
double datsve[] = new double[nn + 1];
double tsve[] = new double[nn + 1];
double delta, l, freq, t, real, imag, angle, theory;

n = nn / 2;
l = 6.; // was 20.
delta = l / ((double) n);
printf_("Test of 1-dimensional FFT");
printf_("nn= " + nn + " n= " + n + " delta= " + delta);
fleopen_(2, "ffttst.dat", 'w');

t = 0.;
fprintf_(2, " i   t-l   h(i)");
for (i = 1; i <= nn; ++i) {
   data[i] = .5 * Math.exp(-Math.abs(t - l));
   // data[i] = Covmodl.covmodl_(1,2,0.,0.,t - l,0.,0.,0.,5);
   fprintf_(2, i + "  " + fdble_(t-l, 9, 5) + "  " + fdble_(data[i], 9, 5));
   datsve[i] = data[i];
   tsve[i] = t;
   t += delta;
}

realft_(data, n);

// Time shift back "l" units.  See Brigham (1974, p. 37).

fprintf_(2, "  indx     freq      real       imag       theory");
fprintf_(2," 0      0       " + (delta * data[1]) + " 0     1.");
for (i = 2; i <= n; ++i) {
   freq = ((double) (i - 1)) / (((double) nn) * delta);
   angle = PIT2 * freq * l;
   angle = PI * ((double) (i - 1));
   real = data[2*i-1] * Math.cos(angle);
   real -= data[2*i] * Math.sin(angle);
   imag = data[2*i-1] * Math.sin(angle);
   imag += data[2*i] * Math.cos(angle);

   theory = 1. / (1. + 4. * PI * PI * freq * freq);
   // theory = Math.exp(-PI * PI * freq * freq);
   fprintf_(2, (i - 1) + " " + fdble_(freq, 9 , 5) + " " +
      fdble_(delta * real, 9, 5) + " " +
     fdble_(delta * imag, 9, 5) + " " + fdble_(theory, 9, 5));
}
freq = ((double) (n)) / (((double) nn) * delta);
angle = PIT2 * freq * l;
real = data[2] * Math.cos(angle);
imag = data[2] * Math.sin(angle);
theory = 1. / (1. + 4. * PI * PI * freq * freq);

fprintf_(2, n + " " + fdble_(freq, 9, 5) + " " +
   fdble_(delta * real, 9, 5) + " " + fdble_(delta * imag, 9, 5) + " " +
   fdble_(theory, 9, 5));

fclose_(2, 'w');

iderr_("Test of 1-D FFT finished");
}

// -------------------------------------------------------------------------

static int premarplefft_(int n, int mode, double w_real[],
   double w_imag[]) {

int k, nt, nexp;
double s, c1_real, c1_imag, c2_real, c2_imag;
double cmplxval[] = new double[2];

nexp = 1;
for (;;) {
   nt = ipow_(2, nexp);
   if (n == nt) break;
   ++nexp;
}

s = 8. * Math.atan(1.) / ((double) nt);
c1_real = Math.cos(s);
c1_imag = -Math.sin(s);
if (mode != 0) c1_imag = -1. * c1_imag;
c2_real = 1.;
c2_imag = 0.;
for (k = 0; k < nt; ++k) {
   w_real[k] = c2_real;
   w_imag[k] = c2_imag;
   cmplxmul_(c2_real, c2_imag, c1_real, c1_imag, cmplxval);
   c2_real = cmplxval[0];
   c2_imag = cmplxval[1];
}

return nexp;
}

// ----------------------------------------------------------------------

static void marplefft_(int n, int mode, double t, int nexp,
   double w_real[], double w_imag[], double x_real[], double x_imag[]) {

int mm, ll, k, j, jj, i, kk, nn, nv2, nm1;
double s, c1_real, c1_imag, c2_real, c2_imag;
double cmplxval[] = new double[2];

mm = 1;
ll = n;
for (k = 1; k <= nexp; ++k) {
   nn = ll / 2;
   jj = mm + 1;
   for (i = 1; i <= n; i += ll) {
      kk = i + nn;
      c1_real = x_real[i - 1] + x_real[kk - 1];
      c1_imag = x_imag[i - 1] + x_imag[kk - 1];

      x_real[kk - 1] = x_real[i - 1] - x_real[kk - 1];
      x_imag[kk - 1] = x_imag[i - 1] - x_imag[kk - 1];

      x_real[i - 1] = c1_real;
      x_imag[i - 1] = c1_imag;
   }

   if (nn == 1) continue;

   for (j = 2; j <= nn; ++j) {
      c2_real = w_real[jj - 1];
      c2_imag = w_imag[jj - 1];
      for (i = j; i <= n; i += ll) {
         kk = i + nn;
         c1_real = x_real[i - 1] + x_real[kk - 1];
         c1_imag = x_imag[i - 1] + x_imag[kk - 1];

         cmplxmul_((x_real[i - 1] - x_real[kk - 1]),
                   (x_imag[i - 1] - x_imag[kk - 1]),
                   c2_real, c2_imag, cmplxval);
         x_real[kk - 1] = cmplxval[0];
         x_imag[kk - 1] = cmplxval[1];

         x_real[i - 1] = c1_real;
         x_imag[i - 1] = c1_imag;
      }
      jj = jj + mm;
   }
   ll = nn;
   mm = mm * 2;
}

nv2 = n / 2;
nm1 = n - 1;
j = 1;
for (i = 1; i <= nm1; ++i) {
   if (i < j) {
      c1_real = x_real[j - 1];
      c1_imag = x_imag[j - 1];

      x_real[j - 1] = x_real[i - 1];
      x_imag[j - 1] = x_imag[i - 1];

      x_real[i - 1] = c1_real;
      x_imag[i - 1] = c1_imag;
   }
   k = nv2;
   for (;;) { 
      if (k >= j) break;
      j = j - k;
      k = k / 2;
   }
   j = j + k;
}

if (mode == 0) s = t;
else s = 1. / (t * ((double) n));

for (i = 0; i < n; ++i) {
   x_real[i] *= s;
   x_imag[i] *= s;
}

return;
}

// -------------------------------------------------------------------------

static void ff2d_(int mode, int n1, int n2, double t1, double t2,
   double x_real[][], double x_imag[][], double y_real[][],
   double y_imag[][]) {

int npsd1 = NMSSTP, npsd2 = NMTSTP, n2max = NMTSTP, k, j, nexp1, nexp2;
double w1_real[] = new double[NMSSTP];
double w1_imag[] = new double[NMSSTP];
double z1_real[] = new double[NMSSTP];
double z1_imag[] = new double[NMSSTP];

double w2_real[] = new double[NMTSTP];
double w2_imag[] = new double[NMTSTP];
double z2_real[] = new double[NMTSTP];
double z2_imag[] = new double[NMTSTP];

double temp_real[][] = new double[NMSSTP][NMTSTP];
double temp_imag[][] = new double[NMSSTP][NMTSTP];

nexp1 = premarplefft_(npsd1, mode, w1_real, w1_imag);
nexp2 = premarplefft_(npsd2, mode, w2_real, w2_imag);
 
// Compute the column transforms;
 
for (k = 0; k < n2; ++k) {
   for (j = 0; j < n1; ++j) {
      z1_real[j] = x_real[j][k];
      z1_imag[j] = x_imag[j][k];
   }

   for (j = n1; j < npsd1; ++j) {
      z1_real[j] = 0.;
      z1_imag[j] = 0.;
   }

   marplefft_(npsd1, mode, 1., nexp1, w1_real, w1_imag, z1_real, z1_imag);

   for (j = 0; j < npsd1; ++j) {
      temp_real[j][k] = z1_real[j];
      temp_imag[j][k] = z1_imag[j];
   }
}
 
// Compute the row transforms and then the 2-D FFT.
 
for (k = 0; k < npsd1; ++k) {
   for (j = 0; j < n2; ++j) {
      z2_real[j] = temp_real[k][j];
      z2_imag[j] = temp_imag[k][j];
   }

   for (j = n2; j < npsd2; ++j) {
      z2_real[j] = 0.;
      z2_imag[j] = 0.;
   }

   marplefft_(npsd2, mode, 1., nexp2, w2_real, w2_imag, z2_real, z2_imag);

   for (j = 0; j < npsd2; ++j) {
      y_real[k][j] = z2_real[j];
      y_imag[k][j] = z2_imag[j];
   }
}

return;
}

// -----------------------------------------------------------------------

static void f2dtst_() {

/* Test program of 2-dimensional Fourier Transform.  Tests the function
   exp(.5S^2 + .5T^2).  The Fourier transform is exp(.5f_S^2 + .5f_T^2)
   (Cartwright, M. (1990), "Fourier Methods for Mathematicians,
   Scientists and Engineers," Ellis Horwood, pp. 117-118). */

int i, j, ns, nt, nns, nnt;
double a, b, c, d, f1, f2, ri, rj, realval, imag, fnorm, dif, maxdif,
   s, t, sl, tl, deltas, deltat, angle, theory;
double indat_real[][] = new double[NMSSTP][NMTSTP];
double indat_imag[][] = new double[NMSSTP][NMTSTP];
double datstore_real[][] = new double[NMSSTP][NMTSTP];
double datstore_imag[][] = new double[NMSSTP][NMTSTP];
double trnsfrm_real[][] = new double[NMSSTP][NMTSTP];
double trnsfrm_imag[][] = new double[NMSSTP][NMTSTP];

sl = 18.0;
tl = 18.0;
nns = NMSSTP;
nnt = NMTSTP;
ns = nns / 2;
nt = nnt / 2;
deltas = 2. * sl / ((double) (nns - 1));
deltat = 2. * tl / ((double) (nnt - 1));
s = 0.;
for (i = 0; i < nns; ++i) {
   t = 0.;
   for (j = 0; j < nnt; ++j) {
      a = s - sl;
      b = t - tl;
      indat_real[i][j] = (1. / PIT2) * Math.exp(-.5 * (a * a + b * b));
      indat_imag[i][j] = 0.;
      datstore_real[i][j] = indat_real[i][j];
      datstore_imag[i][j] = indat_imag[i][j];
      t = t + deltat;
   }
   s = s + deltas;
}

ff2d_(0, nns, nnt, deltas, deltat, indat_real, indat_imag,
   trnsfrm_real, trnsfrm_imag);

printf_(" f1       f2      Re(fft2d)      Im(fft2d)   Theory");
for (i = 1; i <= ns; ++i) {
   ri = (double) (i - 1);
   f1 = PIT2 * ri / (((double) nns) * deltas);
   for (j = 1; j <= nt; ++j) {
      rj = (double) (j - 1);
      f2 = PIT2 * rj / (((double) nnt) * deltat);

      /* Let Hs(ms,mt) be the transform returned by ff2d (which is
         assuming the sampling interval is ((0, 2 * sl),(0, 2 * tl))).

         (ms, mt) is the integer pair corresponding to the frequency pair
         (fs, ft).  Note that fs = ms / (deltas * nns) and
                              ft = mt / (deltat * nnt).

         The (sl, tl)-shifted transform is:

         H(fs,ft) = exp(i * PIT2 * (fs * sl + ft * tl)) * Hs(fs,ft)

         and hence

         H(ms,mt) = exp(i * PIT2 * (sl * (ms / (deltas * nns)) +
                    tl * (mt / (deltat * nnt))) * Hs(ms,mt).

         Since sl = nns * deltas / 2 and tl = nnt * deltat / 2, the above is
       
         H(ms,mt) = exp(i * PI * (ms + mt)) * Hs(ms,mt).
                  = (ac - bd) + i(ad + bc)

         where
         a = cos(angle), b = sin(angle),
         c = real(Hs(ms,mt)), d = imag(Hs(ms,mt)).
         and angle = PI * (ms + mt) */

      angle = PI * (ri + rj);
      a = Math.cos(angle);
      b = Math.sin(angle);
      c = deltas * deltat * trnsfrm_real[i - 1][j - 1];
      d = deltas * deltat * trnsfrm_imag[i - 1][j - 1];
      realval = a * c - b * d;
      imag = a * d + b * c;

      theory = Math.exp(-.5 * (f1 * f1 + f2 * f2));

      printf_(fdble_(f1,6,3) + " " + fdble_(f2,6,3) + "  " +
        realval + "   " + imag + "   " + fdble_(theory,9,5));
   }
}

iderr_("End of f2dtst");
}
}
