public class Fmin extends Id {

//-----------------------------------------------------------------

public static double fmin_(int fnm, double a, double b, double tol) {

/* An approximation x to the point where f attains a minimum on
   the interval (ax,bx) is determined.
 
   input..
 
   ax    left endpoint of initial interval
   bx    right endpoint of initial interval
   f     function subprogram which evaluates f(x) for any x
   in the interval (ax,bx)
   tol   desired length of the interval of uncertainty of the final
   result (.ge.0.)
 
   output..
 
   fmin  abcissa approximating the point where  f  attains a
   minimum
 
      The method used is a combination of golden section search and
   successive parabolic interpolation.  Convergence is never much slower
   than that for a Fibonacci search.  If f has a continuous second
   derivative which is positive at the minimum (which is not at ax or
   bx), then convergence is superlinear, and usually of the order of
   about 1.324.
      The function f is never evaluated at two points closer together
   than eps*abs(fmin)+(tol/3), where eps is approximately the square
   root of the relative machine precision.  If f is a unimodal
   function and the computed values of f are always unimodal when
   separated by at least eps*abs(x)+(tol/3), then fmin approximates
   the abcissa of the global minimum of f on the interval (ax,bx) with
   an error less than 3*eps*abs(fmin)+tol.  If f is not unimodal,
   then fmin may approximate a local, but perhaps non-global, minimum to
   the same accuracy.
 This function subprogram is a slightly modified version of the
   Algol 60 procedure localmin given in Richard Brent, Algorithms For
   Minimization Without Derivatives, Prentice-Hall, Inc. (1973).
 
 Another source is the "zeroin" code in Lau (1995), pp. 318-320.
*/

double c,d,e,eps,xm,p,q,r,tol1,t2, u,v,w,fu,fv,fw,fx,x,tol3;

c = .5*(3.0 - Math.sqrt(5.0));
d = 0.0;

// "eps" is machine precision, typically 1.1102e-16.

eps = 1.5e-16;
tol1 = eps + 1.0;
eps = Math.sqrt(eps);

v = a + c*(b-a);
w = v;
x = v;
e = 0.0;
fx = f_(fnm, x);
fv = fx;
fw = fx;
tol3 = tol/3.0;

xm = .5*(a + b);
tol1 = eps*Math.abs(x) + tol3;
t2 = 2.0*tol1;

 // main loop

while (Math.abs(x-xm) > (t2 - .5*(b-a))) {
   p = q = r = 0.0;

   if (Math.abs(e) > tol1) {

      // fit the parabola

      r = (x-w)*(fx-fv);
      q = (x-v)*(fx-fw);
      p = (x-v)*q - (x-w)*r;
      q = 2.0*(q-r);

      if (q > 0.0) {
         p = -p;

      } else {
         q = -q;
      }

      r = e;
      e = d;   
   }

   if ((Math.abs(p) < Math.abs(.5*q*r)) && (p > q*(a-x)) &&
       (p < q*(b-x))) {

      // a parabolic interpolation step

      d = p/q;
      u = x+d;

      // f must not be evaluated too close to a or b

      if (((u-a) < t2) || ((b-u) < t2)) {
         d = tol1;
         if (x >= xm) d = -d;

      }

   } else {

      // a golden-section step

      if (x < xm) {
         e = b-x;

      } else {
         e = a-x;
      }

      d = c*e;
   }

   // f must not be evaluated too close to x

   if (Math.abs(d) >= tol1) {
      u = x+d;

   } else {
      if (d > 0.0) {
         u = x + tol1;

      } else {
         u = x - tol1;
      }
   }

   fu = f_(fnm, u);

   // Update a, b, v, w, and x

   if (fx <= fu) {
      if (u < x) {
         a = u;

      } else {
         b = u;
      }
   }

   if (fu <= fx) {
      if (u < x) {
         b = x;

      } else {
         a = x;
      }

      v = w;
      fv = fw;
      w = x;
      fw = fx;
      x = u;
      fx = fu;

      xm = .5*(a + b);
      tol1 = eps*Math.abs(x) + tol3;
      t2 = 2.0*tol1;

   } else {
      if ((fu <= fw) || (w == x)) {
         v = w;
         fv = fw;
         w = u;
         fw = fu;

         xm = .5*(a + b);
         tol1 = eps*Math.abs(x) + tol3;
         t2 = 2.0*tol1;

      } else if ((fu > fv) && (v != x) && (v != w)) {
         xm = .5*(a + b);
         tol1 = eps*Math.abs(x) + tol3;
         t2 = 2.0*tol1;

      } else {
         v = u;
         fv = fu;

         xm = .5*(a + b);
         tol1 = eps*Math.abs(x) + tol3;
         t2 = 2.0*tol1;
      }
   }
}

return x;
}   

// ------------------------------------------------------------------

static double f_(int fnm, double x) {

double fval = 0.;

if (fnm == 3) {

   /* Find the CDF value at this x-value by integrating the density
      function from 0 degrees up to "x."  Then, subtract this value
      from the uniformly-generated probability. */

   fval = SMScalcs.hdunifprb - Quad.adquad_(3, 0., x);
   fval /= SMScalcs.hdcnstnt;
}

return fval;
}
}
