public class Forcegrph extends SNA {

/*
Based on the spring embedder algorithm of
Eades, P. (1984), ``A Heuristic for Graph Drawing,''
{\em Congressus Numerantium,} 42:149-160.

At convergence (about 100 loops), nodes that are within 2 links of
each other may be quite close to each other.  A cheap solution is to
increase the repulsive force on just pairs that are no more than 2
links apart for just the last 20 loops.

Idea: Create a hierarchy of nodes by partitioning nodes into 20
levels based on each node's eigenvector centrality.  Then, use
Pltgrph.java to plot the graph.  This way, the hierarchy of middlemen
will be more apparent.  And, eigenvector centrality is related to
the importance or connectedness of a player.
*/

/* srestlngth is the spring rest length,
   ks is the spring constant,
   kr is the repulsive force constant,
   ttosratio is the ratio of kr to ks
   deltat is the time step. */

static int nmnodes;

static double xshift = .01, yshift = .01, bigdist = 8.,
   srestlngth = 1.0,
   ks = 1.,
   kr = .4,
   deltat = .01,
   maxdispsqrd = 0.;

static double nodedegree[] = new double[TNMNDS];
static double nodexpos[] = new double[TNMNDS];
static double nodeypos[] = new double[TNMNDS];
static double nodexforce[] = new double[TNMNDS];
static double nodeyforce[] = new double[TNMNDS];

// ------------------------------------------------------------------

public static void forcegrph_(int netindex, String plotfilename) {

boolean makeheavy = false;

String nodelabel = "none";

int i, j, nmloops, leftnode, rightnode, prntintrvl = 0, count = 0, llx, lly,
urx, ury;

double radius, scale = 1.1, avedist = 0., textsze,
   xmin = 10., ymin = 10., xmax = -10., ymax = -10.,
   newxmin = 10., newymin = 10., newxmax = -10., newymax = -10.;

nmnodes = SNA.nmnodes[netindex];

// This algorithm is approximately O(n^2) convergent so;

nmloops = nmnodes * nmnodes / 2;
if (nmnodes < 20) {
   nmloops = 500;
}

// Compute initial positions.

initialpos_();

// Compute each node's degree.

degree_(nmnodes, prednmlinks, predlink, dumvec);
for (i = 0; i < nmnodes; ++i) {
   nodedegree[i] = dumvec[i];
}

// Perform iterative move steps.

prntintrvl = nmloops / 10;
for (i = 0; i < nmloops; ++i) {
   avedist = onemove_();
   ++count;
   if (count == prntintrvl) {
      // printf_("forcegrph: i= " + i + " avedist= " + avedist);
      count = 0;
   }
}

// Scale to the unit box.

for (i = 0; i < nmnodes; ++i) {
   if (xmax < nodexpos[i]) {
      xmax = nodexpos[i];
   }
   if (ymax < nodeypos[i]) {
      ymax = nodeypos[i];
   }
   if (xmin > nodexpos[i]) {
      xmin = nodexpos[i];
   }
   if (ymin > nodeypos[i]) {
      ymin = nodeypos[i];
   }
}

for (i = 0; i < nmnodes; ++i) {
   nodexpos[i] = (nodexpos[i] - xmin) / (xmax - xmin);
   nodeypos[i] = (nodeypos[i] - ymin) / (ymax - ymin);
   nodexpos[i] *= scale;
   nodeypos[i] *= scale;
   nodexpos[i] += xshift;
   nodeypos[i] += yshift;
}

for (i = 0; i < nmnodes; ++i) {
   if (newxmax < nodexpos[i]) {
      newxmax = nodexpos[i];
   }
   if (newymax < nodeypos[i]) {
      newymax = nodeypos[i];
   }
   if (newxmin > nodexpos[i]) {
      newxmin = nodexpos[i];
   }
   if (newymin > nodeypos[i]) {
      newymin = nodeypos[i];
   }
}

/* Open PostScript output file and place PostScript procedures library
   in it.  Scale and translate as necessary. */

fleopen_(2, plotfilename, 'w');

/*
llx = 0; // was 120
lly = 0; // was 120

urx = 5 + (int) (300. * (newxmax - newxmin)); // was 550
ury = 5 + (int) (400. * (newymax - newymin)); // was 500
printf_("forcegrph: newxmin= " + newxmin + " newxmax= " + newxmax +
   "newymin= " + newymin + " newymax= " + newymax +
   " urx= " + urx + " ury= " + ury);
*/

llx = 120;
lly = 120;
urx = 550;
ury = 500;

Pltgrph.pslib_(2, llx, lly, urx, ury);

//fprintf_(2, "250 250 scale" + "\n.45 .4 translate");
fprintf_(2, "300 300 scale" + "\n.45 .9 translate");

/* Get font.  Set character size to 30% of box height so that two
   lines of text can be included.  Finally, set line width. */

fprintf_(2, "/Times-Roman findfont");
textsze = 0.5 / (double) nmnodes; // was .414, then 2.0

/* For a large network:
if (textsze > .015) {
   textsze = .015;
}
*/

strng = textsze + " scalefont setfont";
fprintf_(2, strng);
fprintf_(2, ".0001 setlinewidth");

// Plot links.  Use a heavy line to plot added links.

for (i = 0; i < prednmlinks; ++i) {
   leftnode = predlink[i][0];
   rightnode = predlink[i][1];

   makeheavy = false;
   for (j = 0; j < nmnewlinks; ++j) {
      if ((leftnode == newlink[j][0] && rightnode == newlink[j][1]) ||
          (leftnode == newlink[j][1] && rightnode == newlink[j][0])) {
         makeheavy = true;
	 break;
      }
   }
   if (makeheavy) {
      fprintf_(2, ".005 setlinewidth");
   }
   strng = fdble_(nodexpos[leftnode - 1], 7, 3) + " " +
	   fdble_(nodeypos[leftnode - 1], 7, 3) + " " +
           fdble_(nodexpos[rightnode - 1], 7, 3) + " " +
	   fdble_(nodeypos[rightnode - 1], 7, 3) + " line";
   fprintf_(2, strng);
   if (makeheavy) {
      fprintf_(2, ".0001 setlinewidth");
   }
}

// Plot nodes.

radius = 0.5 / (double) nmnodes; // was 5.88, then 3.0
if (nmnodes > 20 && radius > .02) {
   radius = .02;
}
for (i = 0; i < nmnodes; ++i) {
   if (prednodelevel[i] == 2) {
      nodelabel = playerID[i].replace('h', 'H');

   } else {
      nodelabel = playerID[i];
   }
   nodelabel = name[netindex][i];
   printf_("forcegrph: i= " + i + " nodelabel= " + nodelabel);

   strng = fdble_(nodexpos[i], 7, 3) + " " +
	   fdble_(nodeypos[i], 7, 3) + " " +
	   fdble_(radius, 7, 3) +
         " (" + nodelabel + ")";
   fprintf_(2, strng);
   fprintf_(2, "crc1txt");
}

// Show the page.

fprintf_(2, "showpage");
fprintf_(2, "%%Trailer");
fprintf_(2, "%%EOF");

// Close PostScript output file.

fclose_(2, 'w');
}

// ------------------------------------------------------------------

public static void initialpos_() {

// Computes initial positions.

int i;

double angle, incrmnt;

angle = 0.;
incrmnt = 2. * 3.14159 / (double) nmnodes;
for (i = 0; i < nmnodes; ++i) {
   nodexpos[i] = xshift + 10. * Rndm.rndm1_(0, 0);
   nodeypos[i] = yshift + 10. * Rndm.rndm1_(0, 0);
   /*
   nodexpos[i] = xshift + 10. * Math.cos(angle);
   nodeypos[i] = xshift + 10. * Math.sin(angle);
   */
   angle += incrmnt;
}
}

// ------------------------------------------------------------------

public static double onemove_() {

// Computes one move of each node.

int i, j, i1, i2;

double dx, dy, dist, distsqrd, force, fx, fy, dispsqrd, s, avedist = 0.;

// Initialize net forces.

for (i = 0; i < nmnodes; ++i) {
   nodexforce[i] = 0.;
   nodeyforce[i] = 0.;
}

/* Repulsion between all pairs.  ForceAtlas2 uses pow(degi*degj),1)
   (no root).  Use of root between .3 and .4 helps on a highly
   connected network.  If there is a highly-connected inner core of
   nodes, the repulsion force on singly-connected nodes can be very
   strong even at large distances causing the network to
   excessively pull apart.  One way to mitigate this effect is to
   attenuate any force from a node farther away that 80% of the
   maximum separation distance. */

for (i1 = 0; i1 < nmnodes - 1; ++i1) {
   for (i2 = i1 + 1; i2 < nmnodes; ++i2) {
      dx = nodexpos[i2] - nodexpos[i1];
      dy = nodeypos[i2] - nodeypos[i1];
      if (dx != 0 || dy != 0) {
         distsqrd = dx * dx + dy * dy;
         dist = Math.sqrt(distsqrd);

	 // force = kr / distsqrd;
	 force = kr * Math.pow((nodedegree[i1] * nodedegree[i2]), .4);
	 force /= distsqrd;

	 if (dist > bigdist) {
            force *= .1;
	 }

	 fx = force * dx / dist;
	 fy = force * dy / dist;
	 nodexforce[i1] -= fx;
	 nodeyforce[i1] -= fy;
	 nodexforce[i2] += fx;
	 nodeyforce[i2] += fy;
      }
   }
}

// Spring force between adjacent pairs.

for (i = 0; i < prednmlinks; ++i) {
   i1 = predlink[i][0] - 1;
   i2 = predlink[i][1] - 1;
   dx = nodexpos[i2] - nodexpos[i1];
   dy = nodeypos[i2] - nodeypos[i1];
   if (dx != 0. || dy != 0.) {
      dist = Math.sqrt(dx * dx + dy * dy);
      force = ks * (dist - srestlngth);
      /*
      if ((nodedegree[i1] > 5. && nodedegree[i2] > 1.) ||
          (nodedegree[i1] > 1. && nodedegree[i2] > 5.)) {
         force *= .5;
      }
      */
      fx = force * dx / dist;
      fy = force * dy / dist;
      nodexforce[i1] += fx;
      nodeyforce[i1] += fy;
      nodexforce[i2] -= fx;
      nodeyforce[i2] -= fy;
   }
}

// Update positions.

for (i = 0; i < nmnodes; ++i) {
   dx = deltat * nodexforce[i];
   dy = deltat * nodeyforce[i];

   avedist += Math.sqrt(dx * dx + dy * dy);
   /*
   dispsqrd = dx * dx + dy * dy;
   if (dispsqrd > maxdispsqrd) {
      s = Math.sqrt(maxdispsqrd / dispsqrd);
      dx *= s;
      dy *= s;
   }
   */
   nodexpos[i] += dx;
   nodeypos[i] += dy;
   /*
   if (nodexpos[i] < 0.) {
      nodexpos[i] = 0.;
   }
   if (nodexpos[i] > 10.) {
      nodexpos[i] = 10.;
   }
   if (nodeypos[i] < 0.) {
      nodeypos[i] = 0.;
   }
   if (nodeypos[i] > 10.) {
      nodeypos[i] = 10.;
   }
   */
}
avedist /= ((double) nmnodes);

return (avedist);
}
}
