class Interdictutils extends Interdict {

static int index[] = new int[Id.NMRTEPTS];

static double dist[] = new double[Id.NMRTEPTS];

// --------------------------------------------------------

static double distptline_(double pt0[], double pt1[], double pt2[]) {

/* Computes the perpendicular distance between the point pt0 and
   the line defined by the two end-points, pt1 and pt2.  From
   http://mathworld.wolfram.com/Point-LineDistance2-Dimensional.html */

double xdif, ydif, ptdist, dist;

if (((pt0[0] < Math.min(pt1[0], pt2[0])) &&
     (pt0[1] < Math.min(pt1[1], pt2[1]))) ||
    ((Math.max(pt1[0], pt2[0]) < pt0[0]) &&
     (Math.max(pt1[1], pt2[1]) < pt0[1]))) {
   dist = 1.;
   return dist;
}

xdif = pt2[0] - pt1[0];
ydif = pt2[1] - pt1[1];
ptdist = Math.sqrt(xdif * xdif + ydif * ydif);

if (xdif < 1.e-5 && ydif < 1.e-5) {
   xdif = pt0[0] - pt1[0];
   ydif = pt0[1] - pt1[1];
   dist = Math.sqrt(xdif * xdif + ydif * ydif);

} else {
   dist = Math.abs((xdif * (pt1[1] - pt0[1])) - (ydif * (pt1[0] - pt0[0])));
   dist /= ptdist;
}

return dist;
}

// --------------------------------------------------------

static boolean cross_(double Ax, double Ay, double Bx, double By,
   double Cx, double Cy, double Dx, double Dy) {

/* Determines if two line segments cross or not.  From a webpage by
   Darel Finley (http://alienryderflex.com/intersect/)

   Determines the intersection point of the line segment defined by
   points A and B with the line segment defined by points C and D.
   Returns "true" if the intersection point was found, and stores
   that point in X,Y.
   Returns "false" if there is no determinable intersection point, in
   which case X,Y will be unmodified.
  
   Returns "false" if segments are colinear even if they overlap. */

double X, Y, distAB, theCos, theSin, newX, ABpos;

// Fail if either line segment is zero-length.

if (Ax==Bx && Ay==By || Cx==Dx && Cy==Dy) {
   return false;
}

//  Fail if the segments share an end-point.

if (Ax==Cx && Ay==Cy || Bx==Cx && By==Cy || Ax==Dx && Ay==Dy ||
    Bx==Dx && By==Dy) {
   return false;
}

// (1) Translate the system so that point A is on the origin.

Bx-=Ax;
By-=Ay;
Cx-=Ax;
Cy-=Ay;
Dx-=Ax;
Dy-=Ay;

//  Discover the length of segment A-B.

distAB=Math.sqrt(Bx*Bx+By*By);

// (2) Rotate the system so that point B is on the positive X axis.

theCos=Bx/distAB;
theSin=By/distAB;
newX=Cx*theCos+Cy*theSin;
Cy  =Cy*theCos-Cx*theSin; Cx=newX;
newX=Dx*theCos+Dy*theSin;
Dy  =Dy*theCos-Dx*theSin; Dx=newX;

//  Fail if segment C-D doesn't cross line A-B.

if (Cy<0. && Dy<0. || Cy>=0. && Dy>=0.) {
   return false;
}

/* (3) Discover the position of the intersection point along line
   A-B. */

ABpos=Dx+(Cx-Dx)*Dy/(Dy-Cy);

// Fail if segment C-D crosses line A-B outside of segment A-B.

if (ABpos<0. || ABpos>distAB) {
   return false;
}

/* (4) Apply the discovered position to line A-B in the original
   coordinate system. */

X=Ax+ABpos*theCos;
Y=Ay+ABpos*theSin;

// Success.

return true;
}

// --------------------------------------------------------

static int readroutes_(String flenme, String prfx, int nmrtepts[],
   double rte[][][], double rtetime[]) {

/* Read observed attacker routes, and potential interdiction patrol
   routes. */

boolean gpx = false;

int i, j, nmrtes = 0, year = 0, month = 0, day = 0, nmpoached = 0;

double lngval = 0., latval = 0.;

// Read the routes file and remove funny characters.

Id.fleopen_(8, flenme, 'r');
Id.fleopen_(9, "dumr.dat", 'w');
do {
   strng = Id.fgetline_(8);
   if (strng != null) {
      strng = strng.replaceAll("<", " ");
      strng = strng.replaceAll(">", " ");
      strng = strng.replaceAll("=", " ");
      strng = strng.replaceAll("\"", " ");
//    strng = strng.replaceAll("/", " ");
      Id.fprintf_(9, strng);
   }
} while (strng != null);
Id.fclose_(8, 'r');
Id.fclose_(9, 'w');

/* Now, read routes.  First, set the transformation to the unit
   square. */

lng0 = 0.;
lat0 = 0.;
lngscle = 1.;
latscle = 1.;

Id.fleopen_(8, "dumr.dat", 'r');

if (!gpx) {
   Id.fgetline_(8);
   Id.fgetline_(8);

   do {
      Id.fgetstrng_(8);
      rtetime[nmrtes] = Id.fgetdble_(8);
      Id.fgetint_(8);
      nmpoached = Id.fgetint_(8);

      nmrtepts[nmrtes] = Id.fgetint_(8);
      Id.fgetstrng_(8);
      Id.fgetint_(8);
      for (i = 0; i < nmrtepts[nmrtes]; ++i) {
         rte[nmrtes][i][0] = Id.fgetdble_(8);
         rte[nmrtes][i][1] = Id.fgetdble_(8);
      }

      /* Only read-in attacker routes that resulted in a poaching event.
         Use this filter only when making an "assess" run to evaluate
	 the interdiction patrol route tool's performance. */

      if (prfx.equals("a") && nmpoached == 0) {
         continue;
      }
      
      // Only read-in attacker routes that have more than 3 points.

      if (nmrtepts[nmrtes] < 2) { // was 4
         continue;
      }

      ++nmrtes;
      if (nmrtes >= Id.NMRTES) {
         Id.iderr_("readroutes: nmrtes= " + nmrtes);
      }
   } while (!Id.checkeof_(8));

} else {

   // Read a GPX route file.

   READRTES: for (;;) {
      do {
         if (Id.checkeof_(8)) {
            break READRTES;
         }
         strng = Id.fgetstrng_(8);
      } while (!strng.equals("rte"));

      // Read this route.

      i = 0;
      for (;;) {
         strng = Id.fgetstrng_(8);
         if (strng.equals("lon")) {
            lngval = Id.fgetdble_(8);
            rte[nmrtes][i][0] = (lngval - lng0) / lngscle;

            Id.fgetstrng_(8);
            latval = Id.fgetdble_(8);
            rte[nmrtes][i][1] = (latval - lat0) / latscle;

            ++i;

         } else if (strng.equals("time")) {
	    strng = Id.fgetstrng_(8);
            year = Integer.parseInt(strng);
 
	    strng = Id.fgetstrng_(8);
            month = -Integer.parseInt(strng);

	    strng = Id.fgetstrng_(8);
            day = -Integer.parseInt(strng);

	    rtetime[nmrtes] = ((double) year) +
               Textutils.nmdays_(0, year, month, day) / 365.;

            strng = Id.fgetstrng_(8);
         
         } else if (strng.equals("/rte")) {
            break;
         }
      }
      nmrtepts[nmrtes] = i;
      ++nmrtes;

      if (nmrtes > Id.NMRTES) {
         Id.iderr_("readroutes: nmrtes= " + nmrtes);
      }
   }
}
Id.fclose_(8, 'r');

Id.printf_("readroutes: nmrtes= " + nmrtes);
return nmrtes;
}

// ---------------------------------------------------------

static void writeroute_(double solt, double lng0, double lat0,
   double lngscle, double latscle) {

/* Randomly selects an interdiction patrol route using the optimal
   probability distribution over these routes -- and then writes this
   route in GPX format to a user-named file. */

int l, i, rte;

double lngval = 0., latval = 0.;

String datetime;

Id.printf_("wrteroutes: solt= " + solt + " Optimal attacker routes:");
for (l = 1; l <= nmtypes; ++l) {
   Id.printf_("   Type= " + l + " Route= " + chosenattrte[l - 1]);
}

Id.printf_("\nOptimal interdiction patrol route probabilities:");
for (i = 1; i <= nmintrrtes; ++i) {
   Id.printf_("  " + i + " " +
      Id.fdble_(intrrteprb[i - 1], 5, 3));
}

Id.printf_(
   "\nRandomly selecting a patrol route from above distribution:");

rte = Rndm.discrte_(0, intrrteprb);
Id.printf_("   execute interdiction patrol route " + rte + "\n" +
               "   written on file \"interdict-patrol-route.gpx\"");

datetime = "2015";
Id.fleopen_(10, "interdict-patrol-route.gpx", 'w');
strng = "<?xml version=\"1.0\" encoding=\"UTF-8\" standalone=\"yes\"?>\n" +
        "<gpx>\n<metadata>\n<name>Interdiction Patrol Route</name>\n" +
	"</metadata>\n<rte>\n<name>Interdiction Patrol Route</name>\n";
Id.fprintf_(10, strng);
for (i = 0; i < nmintrrtepts[rte - 1]; ++i) {
   lngval = lng0 + lngscle * intrrte[rte - 1][i][0];
   latval = lat0 + latscle * intrrte[rte - 1][i][1];
   strng = "<rtept lon=\"" + Id.fdble_(lngval, 7, 3) + "\" lat=\"" +
           Id.fdble_(latval, 7, 3) + "\">\n" +
	   "<time> " + datetime + " </time>\n" +
	   "<name> Position " + (i + 1) + " </name>\n" +
	   "</rtept>";
   Id.fprintf_(10, strng);
}
Id.fprintf_(10, "</rte>\n</gpx>");
Id.fclose_(10, 'w');
}

// --------------------------------------------------------

static double spatialdensest_(int n, double data[][], double pt[]) {

/* Estimates the spatial density function of a spatial point process at
   the location "pt" using a local, m nearest neighbor volumetric
   nonparametric density estimator, see Thompson and Tapia (1990, p. 179).
*/

int i, m = 0;

double fraction = .04, radius, volume, pdfest = 0., smllstvol = 1.e-5,
   cnstnt = 0.;

/* Set the constant to define the maximum value that the
   un-normalized density estimator can take on. */

cnstnt = fraction / smllstvol;

m = (int) Math.round(fraction * ((double) n));

// Find the closest m observations to pt.

for (i = 0; i < n; ++i) {
   dist[i] = Clstr.dist_(2, pt, data[i]);
   index[i] = i;
}

// Sort these distances.

Idsort.idsort_(dist, index, 1, n);

/* Find the m^th nearest neighbor hypersphere radius, r_m(pt) which
   is simply the distance from pt to the m^th closest point. */

radius = dist[m - 1];

// Compute the volume of this hypersphere.

volume = Id.PI * radius * radius;

// Compute density estimate.

if (volume <= smllstvol) {
   pdfest = 1.;

} else {
   pdfest = (fraction / volume) / cnstnt;
}
return pdfest;
}
}
