class SDEcalcs extends Ecosyscalcs {

/* Solves a system of stochastic differential equations one time step
   forward. */

static double x0[][] = new double[NMTHRDS][MAXNMSDE];
static double bk[][] = new double[NMTHRDS][MAXNMSDE];
static double upsilon[][] = new double[NMTHRDS][MAXNMSDE];
static double alpha[][] = new double[NMTHRDS][MAXNMSDE + 3];
static double beta[] = new double[MAXNMSDE];
static double sdesols[][][][][] =
                 new double[NMTHRDS][TNMVALS][SIMSZE][TNMNDS][2];

// --------------------------------------------------------------------

static void sdesol_(int threadnm, double nodevals[][][],
   boolean fixedvars[][], double yn[][], int dst) {

/* Performs a single step of the "Explicit Order 1.0 Strong Scheme" of
   Kloeden and Platen (1995, p. 376) for a vector SDE with d variables
   starting with "startvar" thru "endvar."  The algorithm uses diagonal
   noise, i.e., m = d and each equation is disturbed by only its
   associated Wiener process.  Specifically, the routine returns a
   realization of the vector Ito process at the time point "beltmept." */

int i, j, k;

double ak = 0., delwienk = 0., upsilonk = 0., incrmntk = 0., testval;

/* At the first time point only, either read-in parameters or the
   solution from the last time step. */

/*
if (threadnm == 1) {
   printf_("sdesol: beltmept= " + beltmept[threadnm]);
}
*/

if (firsttimepoint[threadnm]) {
   initialsde_(threadnm, nodevals, dst);
}

// Perform the discrete time approximation step.

for (k = 1; k <= idinfo_nmsdes[idnmbrm1]; ++k) {
   ak = a_(threadnm, nodevals, k, beltmept, yn);
   for (j = 1; j <= idinfo_nmsdes[idnmbrm1]; ++j) {
      bk[threadnm][j - 1] = b_(threadnm, j, k, beltmept, yn);

      upsilon[threadnm][j - 1] = yn[threadnm][j - 1] + ak * delta +
         bk[threadnm][j - 1] * sqrtdelta;
   }
   delwienk = sqrtdelta * Rndm.stdnrm_(threadnm);
   incrmntk = ak * delta + bk[threadnm][k - 1] * delwienk +
      (b_(threadnm, k, k, beltmept, upsilon) - bk[threadnm][k - 1]) *
      (delwienk * delwienk - delta) / (2. * sqrtdelta);

   /* Update all SDE variables, enforce a bounded solution, and check for
      NaN, Infinity. */

   if (!fixedvars[idnmbrm1][k - 1]) {

      /* Force birth (k == 2), and death (k == 3) rates to stay at their
         initial values. */
      
      if (k == 2 || k == 3) {
         testval = x0[threadnm][k - 1];

         /*
         if (k == 3) {
            printf_("sdesol: beltmept= " +
               beltmept[threadnm] + " dthrt= " + testval + " incrmntk= " +
               incrmntk + " yn= " + yn[threadnm][3]);
         }
         */

      } else {

         /*if (k == 4) {
         if (k == 1) {
            printf_("sdesol: beltmept= " + beltmept[threadnm] +
               " ak= " + ak + " delta= " + delta +
               " incrmntk= " + incrmntk + " yn= " + yn[threadnm][k - 1]);
         }
         */

         testval = yn[threadnm][k - 1] + incrmntk;
      }

      if (!Double.isInfinite(testval) && !Double.isNaN(testval)) {
         yn[threadnm][k - 1] = testval;
         if (yn[threadnm][k - 1] < 1.e-10) {

	    // Solution has become too small.

	    if (k == 1) {
	       yn[threadnm][k - 1] = 1.;

	    } else if (k == 2 || k == 3) {
	       yn[threadnm][k - 1] = .001;

	    } else if (k == 4) {
	       yn[threadnm][k - 1] = 1.;
	    }
	 }

         if (yn[threadnm][k - 1] > 1.e6) {

	    // Solution has become too big.

	    iderr_("sdesol: solution is too big, threadnm= " +
               threadnm + " k= " + k + " yn= " + yn[threadnm][k - 1]);
         }

      } else {

	 // Quit if an undefined value has been assigned.

         iderr_("sdesol: testval is undefined threadnm= " +
            threadnm + " k= " + k + " yn= " + yn[threadnm][k - 1]);
      }
   }
}
}

//-------------------------------------------------------------------

static double a_(int threadnm, double nodevals[][][], int k,
   double beltmept[], double yn[][]) {

// Evaluates the trend vector.

boolean test = false;

int i;

double val = 0., discountval = 1., trend, y, ratediff = 0.;

if (idinfo_nmsdes[idnmbrm1] == 1) {
   y = yn[threadnm][k - 1];
   val = alpha[threadnm][1] * y * (1. -  y / alpha[threadnm][0]);
   return val;
}

if (k == 1) {

   // Herbivore count.

   y = yn[threadnm][0];

   // Herbivore model.

   val = alpha[threadnm][1] * y * (1. -  y / alpha[threadnm][0]);

} else if (k == 2 || k == 3) {

   // Birth rate (k=2), and death rate (k=3).

   y = 2. * yn[threadnm][k - 1] - 1.;
   val = -.5 * (alpha[threadnm][k] + beta[k - 1] *
      beta[k - 1] * y) * (1. - y * y);

} else if (k == 4) {

   /* Predator abundance, e.g. cheetah or tiger abundance.  Wide
      differences in the birth and death rates can
      cause the solution to yn[threadnm][3] (Predator count) to blow up.
      If this is happening or the solution has very high variance, try
      forcing this rate difference to be zero as a first try at fixing
      the problem.

      If the carrying capacity is close to predator abundance, abundance
      will be almost completely a function of prey abundance.  In this
      situation,  the birth and death rates will
      have little effect on predator abundance.  To make abundance be
      sensitive to these rates, make the carrying capacity be at least
      four times the number of predators (predator abundance).
   */

   discountval = 1. - Math.pow(alpha[threadnm][4],
                               alpha[threadnm][5] * yn[threadnm][3]);
   trend = yn[threadnm][1] * discountval * yn[threadnm][3];
   trend -= yn[threadnm][2] * yn[threadnm][3];

   // Get the current Carrying Capacity variable value.

   i = Getmodlutils.getndnm_("CarCap");
   alpha[threadnm][6] = nodevals[threadnm][1][i - 1];

   if (alpha[threadnm][6] < 1.e-4) {
      alpha[threadnm][6] = 1.e-4;
   }

   ratediff = yn[threadnm][1] - yn[threadnm][2];
   if (Math.abs(ratediff) > .95) {

      printf_("a_: beltmept= " + beltmept[threadnm] +
         " ratediff= " + ratediff);

      if (ratediff < 0.) {
	 ratediff = -.95;

      } else {
	 ratediff = .95;
      }
   }

   /* Counter-intuitively, a negative trend is caused by the
      birthrate being slightly higher than the deathrate. */

   trend -= ratediff * yn[threadnm][3] * yn[threadnm][3] /
	    alpha[threadnm][6];

   /*
   printf_("alpha= " + alpha[threadnm][6] + " trend= " + trend +
           " ratediff= " + ratediff);
   */
 
   if (Double.isInfinite(trend) || Double.isNaN(trend)) {
      printf_("Ecosyscalcs.a: beltmept= " + beltmept[threadnm] +
         " trend= " + trend +
	 "\n   y2= " + yn[threadnm][1] + " yn3= " + yn[threadnm][2] +
	 " y4= " + yn[threadnm][3] + "\n   a6= " + alpha[threadnm][6]);
      iderr_("a: trend= " + trend);
   }

   /*
   printf_("Ecosyscalcs.a: beltmept= " + beltmept[threadnm] +
      " trend= " + trend +
      "\n   y2= " + yn[threadnm][1] + " yn3= " + yn[threadnm][2] +
      " y4= " + yn[threadnm][3] + "\n   a6= " + alpha[threadnm][6]);
   */

   val = trend;
}
return val;
}

//------------------------------------------------------------------

static double b_(int threadnm, int i, int j, double beltmept[],
   double yn[][]) {

// Evaluates the (i, j)^th component of the diffusion matrix.

double val = 0., y;

if (i != j) {
   val = 0.;

} else if (i == 1) {
   val = beta[0];

} else if (i == 2 || i == 3) {
   y = 2. * yn[threadnm][i - 1] - 1.;
   val = beta[i - 1] * (1. - y * y);

} else if (i == 4) {
   val = beta[3];
}
return val;
}

//------------------------------------------------------------------

static void initialsde_(int threadnm, double nodevals[][][], int dst) {

/* Initialize.  Note that the parameters of an SDE
   are not functions of the first parent, Time.  Also, parent values
   are found explicitly.

   For the Cheetah ID, the parameters and parents are as follows:
   Birth rate, f_t parameters: f_0, alpha_f, beta_f
   parents:    Time, PPtd (in this order)

   Death rate, r_t parameters: r_0, alpha_r, beta_r
   parents:    Time, Hunt

   Count, N_t parameters:      N_0, P, c, beta_N
   parents:    Time, BirthR, DthR, CarCap */

int i, k, rgn = 0, dstm1 = dst - 1;

double birthrate = 0.;

// Get region value.

rgn = Getmodlutils.getndnm_(regionnode);
if (rgn < 1) {
   iderr_("initialsde: no region node.");
}
rgnval[threadnm] = (int) nodevals[threadnm][0][rgn - 1];

if (idinfo_nmsdes[idnmbrm1] == 1) {

   /* Set Herbivore number parameters, B_0, k_0, sigma.  B_0 = k_0 and
      is stored in alpha[0], r is stored in alpha[1], and sigma is stored
      in beta[0].

   k = Getmodlutils.getndnm_("HrbvrAb");
   if (initialIntridsTime) {
   */
      /* Load initial values of the time-stepping functions if this is
         the initial time.  Otherwise, use the values stored in
	 "sdesols" that correspond to this region and realization
	 (indexed by "rgnval" and "idenom[threadnm]", respectively).

      alpha[threadnm][0] = 10000.;
      x0[threadnm][0] = .9 * alpha[threadnm][0];
      nodevals[threadnm][0][k - 1] = x0[threadnm][0];
      nodevals[threadnm][1][k - 1] = x0[threadnm][0];

   } else {
      nodevals[threadnm][0][k - 1] = sdesols[threadnm]
	                                [rgnval[threadnm] - 1]
	                                [idenom[threadnm]][k - 1][dstm1];
      nodevals[threadnm][1][k - 1] = sdesols[threadnm]
	                                [rgnval[threadnm] - 1]
	                                [idenom[threadnm]][k - 1][dstm1];
      x0[threadnm][0] = nodevals[threadnm][1][k - 1];
   } 

   prntvals[threadnm][1] =
      nodevals[threadnm][0][Getmodlutils.getndnm_("PchPrs") - 1];
   prntvals[threadnm][2] = 3.;
   alpha[threadnm][1] = condprb[idnmbrm1][k - 1][0][0]
	         [(int) prntvals[threadnm][1] - 1]
                 [(int) prntvals[threadnm][2] - 1][0][0][0][dstm1];
   beta[0] = condprb[idnmbrm1][k - 1][1][0]
	         [(int) prntvals[threadnm][1] - 1]
                 [(int) prntvals[threadnm][2] - 1][0][0][0][dstm1];
   */

   /* Set Herbivore number parameters, k (herbivore carrying capacity),
      ratediff (difference in herbivore birth and death rates), and
      sigma.  k is stored in alpha[0], and ratediff is in alpha[1].
      sigma is stored in beta[0]. */

   k = Getmodlutils.getndnm_("HrbvrAb");

   prntvals[threadnm][2] = nodevals[threadnm][0]
	                      [Getmodlutils.getndnm_("PchPrs") - 1];
   alpha[threadnm][0] = condprb[idnmbrm1][k - 1][0][0]
                 [(int) prntvals[threadnm][2] - 1][0][0][0][0][dstm1];
   alpha[threadnm][1] = condprb[idnmbrm1][k - 1][1][0]
                 [(int) prntvals[threadnm][2] - 1][0][0][0][0][dstm1];
   beta[0] = condprb[idnmbrm1][k - 1][2][0]
                 [(int) prntvals[threadnm][2] - 1][0][0][0][0][dstm1];

   if (initialIntridsTime || (nmids > 1 && Intridslve.tindex < 3)) {

      /* Initial number of herbivores needs to be (typically) less than
	 alpha_0, say 60% of alpha0.  The equation will deliver a constant
	 solution if x0 = alpha_0. */

      x0[threadnm][0] = .8 * alpha[threadnm][0];

      /*
      printf_("sdesol: threadnm= " + threadnm + " beltmept= " +
         beltmept[threadnm] + " initial herbivore x0= " + x0[threadnm][0]);
      */

   } else {
      x0[threadnm][0] = sdesols[threadnm][rgnval[threadnm] - 1]
	                       [idenom[threadnm]][k - 1][dstm1];
   }
   nodevals[threadnm][0][k - 1] = x0[threadnm][0];
   nodevals[threadnm][1][k - 1] = x0[threadnm][0];

   /* Initialize the predator's Carrying Capacity variable which depends
      linearly on Herbivore abundance (HrbvrAb). */

   update_(threadnm, nodevals, Getmodlutils.getndnm_("CarCap"), dst);
}

// SDE system for a predator.

if (idinfo_nmsdes[idnmbrm1] == 4) {

   // Set Birth Rate parameters, f_0, alpha_f, beta_f.

   k = Getmodlutils.getndnm_("BirthR");
   prntvals[threadnm][1] = nodevals[threadnm][0]
	                      [Getmodlutils.getndnm_("PPtd") - 1];

   if (initialIntridsTime || (nmids > 1 && Intridslve.tindex < 3)) {
      x0[threadnm][1] = condprb[idnmbrm1][k - 1][0][0]
	                   [(int) prntvals[threadnm][1] - 1][0][0][0]
			   [0][dstm1];

   } else {
      x0[threadnm][1] = sdesols[threadnm][rgnval[threadnm] - 1]
	                       [idenom[threadnm]][k - 1][dstm1];
   }
   nodevals[threadnm][0][k - 1] = x0[threadnm][1];
   nodevals[threadnm][1][k - 1] = x0[threadnm][1];
   birthrate = x0[threadnm][1];

   alpha[threadnm][2] = condprb[idnmbrm1][k - 1][1][0]
	                   [(int) prntvals[threadnm][1] - 1][0][0][0]
			   [0][dstm1];
   beta[1] = condprb[idnmbrm1][k - 1][2][0]
	                  [(int) prntvals[threadnm][1] - 1][0][0][0]
			  [0][dstm1];
   
   // Next, set Death Rate parameters, r_0, alpha_r, beta_r.

   k = Getmodlutils.getndnm_("DthR");
   prntvals[threadnm][1] = nodevals[threadnm][0]
	                      [Getmodlutils.getndnm_("PchPrs") - 1];

   if (initialIntridsTime || (nmids > 1 && Intridslve.tindex < 3)) {
      x0[threadnm][2] = condprb[idnmbrm1][k - 1][0][0]
	                   [(int) prntvals[threadnm][1] - 1][0][0][0]
			   [0][dstm1];

      /*
      printf_("initialsde: initial=true tindex= " + Intridslve.tindex +
         " HrbAb= " + x0[threadnm][0] +
         " birthrate= " + birthrate + " pchprs= " + prntvals[threadnm][1] +
         " r0= " + x0[threadnm][2]);
      */

   } else {
      x0[threadnm][2] = sdesols[threadnm][rgnval[threadnm] - 1]
	                       [idenom[threadnm]][k - 1][dstm1];

      /*
      printf_("initialsde: initial=false pchprs= " + prntvals[threadnm][1] +
         " r0= " + x0[threadnm][2]);
      */

   }
   nodevals[threadnm][0][k - 1] = x0[threadnm][2];
   nodevals[threadnm][1][k - 1] = x0[threadnm][2];

   alpha[threadnm][3] = condprb[idnmbrm1][k - 1][1][0]
	         [(int) prntvals[threadnm][1] - 1][0][0][0][0][dstm1];
   beta[2] = condprb[idnmbrm1][k - 1][2][0]
	         [(int) prntvals[threadnm][1] - 1][0][0][0][0][dstm1];

   // Set predator number parameters, N_0, P, c, beta_N

   //k = Getmodlutils.getndnm_("ChthNm");
   k = Getmodlutils.getndnm_("TigerAb");
   if (initialIntridsTime) {
      x0[threadnm][3] = condprb[idnmbrm1][k - 1][0][0][0][0][0][0]
	                   [0][dstm1];
   
   } else {
      x0[threadnm][3] = sdesols[threadnm][rgnval[threadnm] - 1]
	                       [idenom[threadnm]][k - 1][dstm1];
   }
   nodevals[threadnm][0][k - 1] = x0[threadnm][3];
   nodevals[threadnm][1][k - 1] = x0[threadnm][3];

   alpha[threadnm][4] = condprb[idnmbrm1][k - 1][1][0][0][0][0][0]
	                   [0][dstm1];
   alpha[threadnm][5] = condprb[idnmbrm1][k - 1][2][0][0][0][0][0]
	                   [0][dstm1];
   alpha[threadnm][6] = nodevals[threadnm][0]
	                   [Getmodlutils.getndnm_("CarCap") - 1];
   beta[3] = condprb[idnmbrm1][k - 1][3][0][0][0][0][0]
	                  [0][dstm1];
}

// Load initial values into solution vector.

for (i = 0; i < idinfo_nmsdes[idnmbrm1]; ++i) {
   yn[threadnm][i] = x0[threadnm][i];
}
}

//------------------------------------------------------------------

static void test_() {

// Tests the sdesol method.

int i, j, capn = 1000;

double capt = .5, xt, delta, wt, sqrtdelta;

double alpha[][] = new double[1][1];
double beta[] = new double[1];
double x0[][] = new double[1][3];
double yn[][] = new double[1][MAXNMSDE];

alpha[0][0] = .5;
beta[0] = .1;
x0[0][0] = .5;
x0[0][1] = .5;
x0[0][2] = .5;
fleopen_(2, "sde.dat", 'w');

delta = capt / (double) capn;
sqrtdelta = Math.sqrt(delta);
for (i = 0; i < 100; ++i) {
   wt = 0.;
   for (j = 0; j < capn; ++j) {
      wt += sqrtdelta * Rndm.stdnrm_(0);
   }
   xt = (1. + x0[0][0]) * Math.exp(-2. * alpha[0][0] * capt +
           2. * beta[0] * wt);
   xt = (xt + x0[0][0] - 1.) / (xt + 1. - x0[0][0]);

   /* The call to sdesol_() is broken.  Its current form is:

      static void sdesol_(int threadnm, double nodevals[][][],
         boolean fixedvars[][], double yn[][], int dst) {

      The old form that used to work in this test method is:
      sdesol_(tbegin, beltmept, x0, alpha, beta, yn, delta, sqrtdelta); */

   fprintf_(2, (xt - yn[0][0]) + " " + (xt - yn[0][1]) + " " +
      (xt - yn[0][2]));
}
fclose_(2, 'w');
iderr_("Normal termination of Ecosyscalcs.test_()");
}
}
