DROP DATABASE IF EXISTS node2;
CREATE DATABASE node2;
USE node2;

/* ------------- Criminal intelligence tables. --------------- */

create table if not exists players (
   playerID bigint unsigned not null default 0,
   name    varchar(18) not null default 'unknown name',
   town    varchar(15) not null default 'unknown town',
   country varchar(15) not null default 'unknown country',
   primary key(name, country)
) engine=InnoDB;

create table if not exists phones (
   owner varchar(30),
   foreign key (owner) references players(name)
      on delete cascade,
   phone_number int unsigned not null default 000000
) engine=InnoDB;

create table if not exists vehicles (
   owner varchar(30),
   foreign key (owner) references players(name)
      on delete cascade,
   registration_number varchar(10) not null default ''
) engine=InnoDB;

create table if not exists firearms (
   owner varchar(30),
   foreign key (owner) references players(name)
      on delete cascade,
   registration_number varchar(10) not null default ''
) engine=InnoDB;

create table if not exists bank_accounts (
   owner varchar(30),
   foreign key (owner) references players(name)
      on delete cascade,
   account_number int unsigned not null default 000000
) engine=InnoDB;

create table if not exists links (
   from_name varchar(30) not null,
   to_name varchar(30) not null,
   evidence_type varchar(30) not null,
   foreign key (from_name) references players(name)
      on delete cascade,
   foreign key (to_name) references players(name)
      on delete cascade
) engine=InnoDB;

create table if not exists wire_transfers (
   from_account varchar(30) not null,
   foreign key (from_account) references bank_accounts(owner)
      on delete cascade,
   to_account varchar(30) not null,
   foreign key (to_account) references bank_accounts(owner)
      on delete cascade,
   transfer_date date,
   amount int not null default 000000
) engine=InnoDB;

create table if not exists seized_shipments (
   root varchar(30),
   country varchar(30),
   foreign key (root, country) references players(name, country)
      on delete cascade,
   seizure_date date,
   wildlife_product varchar(30) not null default 'unknown product',
   amount int not null default 000000
) engine=InnoDB;

/* ------------ Procedures to create Player IDs. -------------- */

create table if not exists ids (
   id bigint unsigned not null primary key,
   taken tinyint default 0
) engine=INNoDB;

/* Update ids table with random ID numbers.  To assign a new, random, and
   unique id, simply grab a random open one from the IDs table.  Do this
   by grabbing a set of ids and attempting to make the UPDATE with one of
   them. If the update fails, then some other process grabbed that ID.
   That's no biggie, just try another. */

delimiter //
CREATE PROCEDURE randomid()
BEGIN
   DECLARE n INT DEFAULT 0;
   SELECT COUNT(*) FROM players INTO n;
   SET @i=0;
   WHILE @i<n DO 
      set @val = (SELECT id FROM ids WHERE taken=0 LIMIT 1);
      update players set playerID=@val where playerID=0 limit 1;
      UPDATE ids SET taken=1 WHERE id=@val AND taken=0;
      SET @i = @i + 1;
   END WHILE;
End//
delimiter ;
 
/* Any time we run out of IDs, we can generate new ones with this stored
   procedure. */

delimiter //
CREATE PROCEDURE create_ids (create_id_num INT)
BEGIN
   DECLARE CONTINUE HANDLER FOR SQLSTATE '42000' BEGIN END;
   SET @i = 1;
   loop1: LOOP
      INSERT INTO ids (id) VALUES (FLOOR(1 + RAND() * POW(2,63)));
      SET @i = @i + 1;
      IF @i > create_id_num THEN
         LEAVE loop1;
      END IF;
   END LOOP loop1;
END//
delimiter ;

/* Create a trigger that ensures that there are more IDs created when the
   last one is taken. */

delimiter //
CREATE TRIGGER make_additional_ids AFTER UPDATE ON ids FOR EACH ROW
BEGIN
   IF (SELECT COUNT(*) FROM ids WHERE taken = 0) = 0 THEN
      CALL create_ids(1000);
   END IF;
END//
delimiter ;

-- Create 100 random ID values.

call create_ids(100);

/* -------------------- Insert data into database ----------------------- */

insert into players
   (name, town, country) values
   ('p1',  't1', 'c1'), 
   ('p2',  't4', 'c3'), 
   ('p3',  't1', 'c1'), 
   ('p4',  't1', 'c1'), 
   ('p5',  't3', 'c2'), 
   ('p6',  't1', 'c1'), 
   ('p7',  't1', 'c1'), 
   ('p8',  't1', 'c1'), 
   ('p9',  't1', 'c1'), 
   ('p10', 't1', 'c1'), 
   ('p11', 't3', 'c2');

-- Assign IDs to these players.

call randomid();

insert into vehicles
   (owner, registration_number) values
   ('p6',  'lu7'),
   ('p11', 'lu7');

insert into links
   (from_name, to_name, evidence_type) values
   ('p10', 'p9', 'call'), 
   ('p9',  'p7', 'call'), 
   ('p9',  'p8', 'shipment'), 
   ('p7',  'p8', 'shipment'), 
   ('p6',  'p8', 'transfer'), 
   ('p5',  'p6', 'call'), 
   ('p5',  'p4', 'call'), 
   ('p5',  'p1', 'call'), 
   ('p5',  'p2', 'call'), 
   ('p5',  'p3', 'call'), 
   ('p2',  'p3', 'call'), 
   ('p11', 'p8', 'call'), 
   ('p11', 'p1', 'transfer');

select * from players;
select * from vehicles;
select * from links;

/* Create users accounts on this node only.  Note that creating these
   accounts on the logistics node does not create them on any other
   node.  Therefore, each node needs to create all confederation member
   accounts.  But these creation commands need to be commented-out when
   this script is run on the same server that createlogistics.sql is
   run on. */

/*
drop user if exists 'Jay Lee';
create user 'Jay Lee'
   identified by 'passwrd1';
grant select, insert
   on *.*
   to 'Jay Lee';

drop user if exists 'Jeff Lee';
create user 'Jeff Lee'
   identified by 'passwrd2';
grant select, insert
   on *.*
   to 'Jeff Lee';

drop user if exists 'George Lee';
create user 'George Lee'
   identified by 'passwrd3';
grant select, insert
   on *.*
   to 'George Lee';
*/

/* Print user privileges. */

show grants for 'Jay Lee';
show grants for 'Jeff Lee';
show grants for 'George Lee';
