DROP DATABASE IF EXISTS members;
CREATE DATABASE members;
SHOW DATABASES;
USE members;
SELECT DATABASE();
create table if not exists member_information (
   memberID bigint unsigned not null default 0,
   name     varchar(18) not null default 'no name',
   country  varchar(15) not null default 'no country',
   email    varchar(30) not null default 'not known',
   primary key(memberID)
) engine=InnoDB;

create table if not exists node_information (
   nodeID bigint unsigned not null default 0,
   name     varchar(18) not null default 'no name',
   country  varchar(15) not null default 'no country',
   email    varchar(30) not null default 'not known',
   primary key(nodeID)
) engine=InnoDB;

create table if not exists privileges_information (
   privilegeID int unsigned not null default 0,
   privilegename varchar(18) not null default 'none',
   primary key(privilegeID)
) engine=InnoDB;

/* Junction table: node=0 is the member's global authorization.*/

create table if not exists member_authorizations (
   memberID bigint unsigned not null,
   nodeID bigint unsigned not null,
   privilegeID int unsigned not null,
   constraint c1 primary key (memberID, nodeID, privilegeID),
   foreign key (memberID) references member_information (memberID),
   foreign key (nodeID) references node_information (nodeID),
   foreign key (privilegeID) references privileges_information (privilegeID)
) engine=InnoDB;

/* Create a trigger to update the global authorizations table whenever a
   node update's its local authorizations.  Implements the GLAD access
   control system using its "strictly conservative" strategy:
   Procedure SC-strategy (Figure 3 in Castano et al. (1997). */

delimiter //
create trigger update_global_authorizations
after update
on member_authorizations for each row begin

   -- Count the number of nodes.

   set @nmnodes = (select count(nodeID) from node_information);

   -- Delete old global authorizations.

   delete from member_authorizations where nodeID = 0;

   -- Select member privileges that were given to them by all nodes.

   create temporary table n (
   memberID bigint unsigned not null default 0,
   nodeID bigint unsigned not null default 0,
   privilegeID int unsigned not null default 0);

   insert into n
   select memberID, privilegeID, count(m.memberID) as nmgivenpriv
   from member_authorizations m
   group by memberID, privilegeID
   having nmgivenpriv = @nmnodes;

   update n
   set nodeID = 0;

   -- Replace the old global authorizations table with these new ones.

   insert into member_authorizations
   select * from n;
end//

delimiter ;

/* -------------------- Insert data into database ----------------------- */

use members;
insert into member_information
(memberID, name, country, email) values
(01, 'Jay Lee',    'USA',    'jay@organization1.com'),
(09, 'Jeff Lee',   'UK',     'jlee@organization2.com'),
(51, 'George Lee', 'France', 'gl@organization3.com')
;

insert into node_information
(nodeID, name, country, email) values
(01, 'Interpol',   'France', 'dbmanager@interpol.com'),
(02, 'State Dept', 'USA',    'dbmanager@biostate.gov')
;

insert into privileges_information
(privilegeID, privilegename) values
(1, 'select'),
(2, 'insert')
;
select * from member_information;
select * from node_information;
select * from privileges_information;
