/* Updates user privileges on the logistics node and writes an SQL script
   file that is emailed to all nodes so that they can update their user
   privileges.  This script is run by a root on the logistics node when
   he/she receives an email from a node with an attached
   "required_changes.sql" script file that contains changes to the privileges
   of one or more members that this node is requiring to be changed in the
   GLAD authorizations table that is contained in the logistics node's
   "member" database. */

use members;

/*
select m.name, p.privilegename
   from member_information m, member_authorizations g,
      privileges_information p
   where m.memberID = g.memberID and g.nodeID = 0 and
      g.privilegeID = p.privilegeID;

-- Equivalent query.
select m.name, p.privilegename
   from member_information m
   join member_authorizations g on m.memberID = g.memberID
   join privileges_information p on g.privilegeID = p.privilegeID
   where g.nodeID = 0;
*/

/* Revoke data access privileges for all users. */

drop procedure if exists revokeprivs_;

DELIMITER //
CREATE PROCEDURE revokeprivs_()
BEGIN
   DECLARE finished INTEGER DEFAULT 0;
   DECLARE membername varchar(18) DEFAULT "";

   DECLARE currevoke CURSOR FOR 
      select name
      from member_information;

   DECLARE CONTINUE HANDLER FOR NOT FOUND SET finished = 1;

   OPEN currevoke;
   Mainloop: LOOP
      FETCH currevoke INTO membername;

      IF finished = 1 THEN 
         LEAVE Mainloop;
      END IF;

      /* Give the member "select" privilege just so the "revoke" command
         does not error-out. */

      set @quotedname = concat("'",membername,"'");
      set @cmmd = concat('grant select  on *.* to ',@quotedname,";");
      prepare stmt from @cmmd;
      execute stmt;
      select @cmmd;

      /*
      set @cmmd = concat('show grants for ',@quotedname,";");
      prepare stmt from @cmmd;
      execute stmt;
      */

      set @cmmd = concat('revoke all on *.* from ',@quotedname,";");
      prepare stmt from @cmmd;
      execute stmt;
      select @cmmd;

   END LOOP Mainloop;
   CLOSE currevoke;
END//
DELIMITER ;

call revokeprivs_();
flush privileges;

/* Now, set each user's privileges according to the global
   authorizations table that was updated when the query
   "required_changes.sql" was run against this database. */

drop procedure if exists giveprivs_;

DELIMITER //
CREATE PROCEDURE giveprivs_()
BEGIN
   DECLARE finished   INTEGER DEFAULT 0;
   DECLARE membername varchar(18) DEFAULT "";
   DECLARE privname varchar(18) DEFAULT "";

   DECLARE curgive CURSOR FOR 
      select m.name, p.privilegename
         from member_information m, member_authorizations g,
            privileges_information p
         where m.memberID = g.memberID and g.nodeID = 0 and
            g.privilegeID = p.privilegeID;

   DECLARE CONTINUE HANDLER FOR NOT FOUND SET finished = 1;

   OPEN curgive;
   Mainloop: LOOP
      FETCH curgive INTO membername, privname;

      IF finished = 1 THEN 
         LEAVE Mainloop;
      END IF;

      set @quotedname = concat("'",membername,"'");
      set @cmmd = concat('grant ',privname,' on *.* to ',@quotedname,";");
      prepare stmt from @cmmd;
      execute stmt;

      set @cmmd = concat('show grants for ',@quotedname,";");
      prepare stmt from @cmmd;
      execute stmt;

      select " ";
   END LOOP Mainloop;
   CLOSE curgive;
END//
DELIMITER ;

call giveprivs_();
flush privileges;
